import React, { useState, useEffect } from 'react';

function App() {
	const [isLoggedIn, setIsLoggedIn] = useState(false);
	const [showAuth, setShowAuth] = useState(false);
	const [authMode, setAuthMode] = useState('login'); // 'login' or 'register'
	const [authError, setAuthError] = useState('');
	const [authForm, setAuthForm] = useState({ username: '', password: '', role: 'client' });
	const [isAdmin, setIsAdmin] = useState(false);

	const [menu, setMenu] = useState([]);
	const [activeTab, setActiveTab] = useState('Home');
	const [showPopup, setShowPopup] = useState(true);

	useEffect(() => {
		fetch('/api/pages')
			.then(res => res.json())
			.then(data => {
				setMenu(data.map(page => ({ name: page.name, link: `/page/${page.id}` })));
			})
			.catch(() => setMenu([]));
	}, []);



	// Video sources: local, YouTube, live

	// State to check if local video exists
	const [hasLocalVideo, setHasLocalVideo] = useState(false);
	useEffect(() => {
		// Check if local video exists
		fetch('/media/welcome.mp4', { method: 'HEAD' })
			.then(res => {
				setHasLocalVideo(res.ok);
			})
			.catch(() => setHasLocalVideo(false));
	}, []);

	const getVideoContent = (type) => {
		if (type === 'Home') {
			return (
				<>
					<h2 style={{ marginBottom: '1rem' }}>Welcome to alo@laba!</h2>
					<div style={{ position: 'relative', width: '100%', paddingBottom: '56.25%', maxWidth: 800, margin: '0 auto' }}>
						{hasLocalVideo ? (
							<video
								style={{
									position: 'absolute',
									top: 0,
									left: 0,
									width: '100%',
									height: '100%',
									objectFit: 'cover',
									borderRadius: '8px',
								}}
								controls
								autoPlay
								loop
								muted
								playsInline
								src="/media/welcome.mp4"
							>
								Your browser does not support the video tag.
							</video>
						) : (
							<iframe
								src="https://www.youtube.com/embed/dQw4w9WgXcQ?autoplay=1&mute=1"
								title="Welcome Video"
								frameBorder="0"
								allow="autoplay; encrypted-media"
								allowFullScreen
								style={{
									position: 'absolute',
									top: 0,
									left: 0,
									width: '100%',
									height: '100%',
									borderRadius: '8px',
								}}
							></iframe>
						)}
					</div>
				</>
			);
		}
		if (type === 'Live') {
			return (
				<>
					<h2 style={{ marginBottom: '1rem' }}>Live Stream</h2>
					<div style={{ position: 'relative', width: '100%', paddingBottom: '56.25%', maxWidth: 800, margin: '0 auto' }}>
						<iframe
							src="https://www.youtube.com/embed/live_stream?channel=UC4R8DWoMoI7CAwX8_LjQHig&autoplay=1&mute=1"
							title="Live Stream"
							frameBorder="0"
							allow="autoplay; encrypted-media"
							allowFullScreen
							style={{
								position: 'absolute',
								top: 0,
								left: 0,
								width: '100%',
								height: '100%',
								borderRadius: '8px',
							}}
						></iframe>
					</div>
				</>
			);
		}
		return null;
	};

	const popup = showPopup && (
		<div
			style={{
				position: 'fixed',
				top: 0,
				left: 0,
				width: '100vw',
				height: '100vh',
				background: 'rgba(0,0,0,0.4)',
				display: 'flex',
				alignItems: 'center',
				justifyContent: 'center',
				zIndex: 1000,
				padding: '2vw',
			}}
		>
			<div
				style={{
					background: '#fff',
					padding: '3vw',
					borderRadius: '24px',
					minWidth: '340px',
					width: '100%',
					maxWidth: 950,
					textAlign: 'center',
					boxSizing: 'border-box',
				}}
			>
				<div style={{ marginBottom: '1rem', display: 'flex', justifyContent: 'center', gap: '1rem', flexWrap: 'wrap' }}>
					<button
						onClick={() => setActiveTab('Home')}
						style={{
							padding: '0.5rem 1.5rem',
							background: activeTab === 'Home' ? '#222' : '#eee',
							color: activeTab === 'Home' ? '#fff' : '#222',
							border: 'none',
							borderRadius: '4px',
							fontWeight: activeTab === 'Home' ? 'bold' : 'normal',
							cursor: 'pointer',
							minWidth: 100,
						}}
					>
						Home
					</button>
					<button
						onClick={() => setActiveTab('Live')}
						style={{
							padding: '0.5rem 1.5rem',
							background: activeTab === 'Live' ? '#222' : '#eee',
							color: activeTab === 'Live' ? '#fff' : '#222',
							border: 'none',
							borderRadius: '4px',
							fontWeight: activeTab === 'Live' ? 'bold' : 'normal',
							cursor: 'pointer',
							minWidth: 100,
						}}
					>
						Live
					</button>
				</div>
				<div style={{ marginBottom: '1rem', width: '100%' }}>
					{getVideoContent(activeTab)}
				</div>
				<button
					onClick={() => setShowPopup(false)}
					style={{
						padding: '0.5rem 2rem',
						background: '#222',
						color: '#fff',
						border: 'none',
						borderRadius: '4px',
						width: '100%',
						maxWidth: 200,
						margin: '0 auto',
						display: 'block',
					}}
				>
					Close
				</button>
			</div>
		</div>
	);

	// Handle login/register form submit
	const handleAuthSubmit = async (e) => {
		e.preventDefault();
		setAuthError('');
		try {
			if (authMode === 'login') {
				const res = await fetch('/api/login', {
					method: 'POST',
					headers: { 'Content-Type': 'application/json' },
					body: JSON.stringify({ username: authForm.username, password: authForm.password })
				});
				const data = await res.json();
				if (!res.ok) throw new Error(data.error || 'Login failed');
				setIsLoggedIn(true);
				setIsAdmin(data.is_admin);
				setShowAuth(false);
				setAuthForm({ username: '', password: '', role: 'client' });
			} else {
				const res = await fetch('/api/register', {
					method: 'POST',
					headers: { 'Content-Type': 'application/json' },
					body: JSON.stringify({ username: authForm.username, password: authForm.password, is_admin: authForm.role === 'admin' })
				});
				const data = await res.json();
				if (!res.ok) throw new Error(data.error || 'Register failed');
				setAuthMode('login');
				setAuthError('Registration successful! Please login.');
			}
		} catch (err) {
			setAuthError(err.message);
		}
	};

	// Auth modal
	const authModal = showAuth && (
		<div style={{
			position: 'fixed', top: 0, left: 0, width: '100vw', height: '100vh', background: 'rgba(0,0,0,0.4)', zIndex: 2000,
			display: 'flex', alignItems: 'center', justifyContent: 'center',
		}}>
			<div style={{ background: '#fff', padding: '2rem', borderRadius: 12, minWidth: 320, maxWidth: 400, width: '100%' }}>
				<h2 style={{ marginBottom: 16 }}>{authMode === 'login' ? 'Login' : 'Register'}</h2>
				<form onSubmit={handleAuthSubmit}>
					<input
						type="text"
						placeholder="Username"
						value={authForm.username}
						onChange={e => setAuthForm(f => ({ ...f, username: e.target.value }))}
						style={{ width: '100%', marginBottom: 12, padding: 8 }}
						required
					/>
					<input
						type="password"
						placeholder="Password"
						value={authForm.password}
						onChange={e => setAuthForm(f => ({ ...f, password: e.target.value }))}
						style={{ width: '100%', marginBottom: 12, padding: 8 }}
						required
					/>
					{authMode === 'register' && (
						<div style={{ marginBottom: 12 }}>
							<label>
								<input
									type="radio"
									name="role"
									value="client"
									checked={authForm.role === 'client'}
									onChange={() => setAuthForm(f => ({ ...f, role: 'client' }))}
								/> Client
							</label>
							{/* <label style={{ marginLeft: 16 }}>
								<input
									type="radio"
									name="role"
									value="admin"
									checked={authForm.role === 'admin'}
									onChange={() => setAuthForm(f => ({ ...f, role: 'admin' }))}
								/> Admin
							</label> */}
						</div>
					)}
					<button type="submit" style={{ width: '100%', padding: 10, background: '#222', color: '#fff', border: 'none', borderRadius: 4, marginBottom: 8 }}>
						{authMode === 'login' ? 'Login' : 'Register'}
					</button>
				</form>
				<div style={{ color: authError.startsWith('Registration') ? 'green' : 'red', minHeight: 24, marginBottom: 8 }}>{authError}</div>
				<div style={{ textAlign: 'center' }}>
					{authMode === 'login' ? (
						<span>Don't have an account? <button style={{ color: '#222', background: 'none', border: 'none', textDecoration: 'underline', cursor: 'pointer' }} onClick={() => { setAuthMode('register'); setAuthError(''); }}>Register</button></span>
					) : (
						<span>Already have an account? <button style={{ color: '#222', background: 'none', border: 'none', textDecoration: 'underline', cursor: 'pointer' }} onClick={() => { setAuthMode('login'); setAuthError(''); }}>Login</button></span>
					)}
				</div>
				<button onClick={() => setShowAuth(false)} style={{ marginTop: 12, width: '100%', padding: 8, background: '#eee', border: 'none', borderRadius: 4 }}>Close</button>
			</div>
		</div>
	);

	// Admin panel tabs
	const [adminTab, setAdminTab] = useState('users');
	const adminPanel = isAdmin && (
		<div style={{ display: 'flex', margin: '2rem auto', maxWidth: 1100, minHeight: 400, background: '#f9f9f9', borderRadius: 8, boxShadow: '0 2px 8px #0001' }}>
			{/* Sidebar */}
			<div style={{ minWidth: 180, background: '#222', color: '#fff', borderTopLeftRadius: 8, borderBottomLeftRadius: 8, padding: '32px 0', display: 'flex', flexDirection: 'column', alignItems: 'stretch', gap: 0 }}>
				<button onClick={() => setAdminTab('users')} style={{ fontWeight: adminTab === 'users' ? 'bold' : 'normal', background: adminTab === 'users' ? '#fff' : 'transparent', color: adminTab === 'users' ? '#222' : '#fff', border: 'none', borderRadius: 0, padding: '16px 0', cursor: 'pointer', width: '100%' }}>Users</button>
				<button onClick={() => setAdminTab('videos')} style={{ fontWeight: adminTab === 'videos' ? 'bold' : 'normal', background: adminTab === 'videos' ? '#fff' : 'transparent', color: adminTab === 'videos' ? '#222' : '#fff', border: 'none', borderRadius: 0, padding: '16px 0', cursor: 'pointer', width: '100%' }}>Videos</button>
				<button onClick={() => setAdminTab('events')} style={{ fontWeight: adminTab === 'events' ? 'bold' : 'normal', background: adminTab === 'events' ? '#fff' : 'transparent', color: adminTab === 'events' ? '#222' : '#fff', border: 'none', borderRadius: 0, padding: '16px 0', cursor: 'pointer', width: '100%' }}>Events</button>
				<button onClick={() => setAdminTab('settings')} style={{ fontWeight: adminTab === 'settings' ? 'bold' : 'normal', background: adminTab === 'settings' ? '#fff' : 'transparent', color: adminTab === 'settings' ? '#222' : '#fff', border: 'none', borderRadius: 0, padding: '16px 0', cursor: 'pointer', width: '100%' }}>Site Settings</button>
			</div>
			{/* Main content */}
			<div style={{ flex: 1, padding: '32px 40px' }}>
				{adminTab === 'users' && (
					<div>
						<h3>User Management</h3>
						{/* TODO: Add user list, add/edit user forms here */}
						<p>Manage users: add, edit, and list users.</p>
					</div>
				)}
				{adminTab === 'videos' && (
					<div>
						<h3>Video Management</h3>
						{/* TODO: Add video list, add/edit/set active/live video forms here */}
						<p>Manage videos: add, edit, set active/live, and list videos.</p>
					</div>
				)}
				{adminTab === 'events' && (
					<div>
						<h3>Event Management</h3>
						{/* TODO: Add event list, add/edit event forms here */}
						<p>Manage events: add, edit, and list events. Show event registration form as needed.</p>
					</div>
				)}
				{adminTab === 'settings' && (
					<div>
						<h3>Site Settings</h3>
						{/* TODO: Add site settings, add/edit pages and links here */}
						<p>Manage site settings: add/edit pages, links, and other settings.</p>
					</div>
				)}
			</div>
		</div>
	);

	return (
		<div>
			<header style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', padding: '1rem', background: '#222', color: '#fff' }}>
				<div style={{ fontWeight: 'bold', fontSize: '1.5rem' }}>alo@laba</div>
				<nav>
					{menu.map((item) => (
						<a key={item.name} href={item.link} style={{ color: '#fff', margin: '0 1rem', textDecoration: 'none' }}>{item.name}</a>
					))}
				</nav>
				<button
					onClick={() => {
						if (isLoggedIn) {
							setIsLoggedIn(false);
							setIsAdmin(false);
						} else {
							setShowAuth(true);
							setAuthMode('login');
							setAuthError('');
						}
					}}
					style={{ padding: '0.5rem 1rem', background: '#fff', color: '#222', border: 'none', borderRadius: '4px' }}
				>
					{isLoggedIn ? 'Logout' : 'Login'}
				</button>
			</header>
			<main>
				{popup}
				{authModal}
				{isLoggedIn && !isAdmin && (
					<div style={{ textAlign: 'center', margin: '2rem 0' }}>
						<h3>Dashboard</h3>
						<p>Welcome! Here is the main table to view the live or current screen:</p>
						<div style={{ display: 'flex', justifyContent: 'center', marginTop: 32 }}>
							<table style={{ borderCollapse: 'collapse', minWidth: 400, background: '#fff', boxShadow: '0 2px 8px #0001' }}>
								<thead>
									<tr style={{ background: '#222', color: '#fff' }}>
										<th style={{ padding: '12px 24px', border: '1px solid #eee' }}>Title</th>
										<th style={{ padding: '12px 24px', border: '1px solid #eee' }}>Type</th>
										<th style={{ padding: '12px 24px', border: '1px solid #eee' }}>Status</th>
									</tr>
								</thead>
								<tbody>
									<tr>
										<td style={{ padding: '10px 24px', border: '1px solid #eee' }}>Live Stream</td>
										<td style={{ padding: '10px 24px', border: '1px solid #eee' }}>YouTube</td>
										<td style={{ padding: '10px 24px', border: '1px solid #eee', color: 'green', fontWeight: 'bold' }}>Active</td>
									</tr>
									<tr>
										<td style={{ padding: '10px 24px', border: '1px solid #eee' }}>Welcome Video</td>
										<td style={{ padding: '10px 24px', border: '1px solid #eee' }}>Local/YouTube</td>
										<td style={{ padding: '10px 24px', border: '1px solid #eee', color: 'gray' }}>Available</td>
									</tr>
								</tbody>
							</table>
						</div>
					</div>
				)}
				{isAdmin && (
					<div style={{ textAlign: 'center', margin: '2rem 0' }}>
						<h3>Private Content</h3>
						<p>This is private or paid content only visible to logged-in users.</p>
						<div style={{ color: 'green', fontWeight: 'bold' }}>You are an Admin.</div>
					</div>
				)}
				{adminPanel}
			</main>
		</div>
	);
}

export default App;